/*! \file
 * \brief Main camera functions
 * \author Adam Klaptocz (largely based on code by Philippe Rtornaz)
 */
#include <p33FJ256GP506.h>
#include "e_omni_ports.h"
#include "e_po3030k.h"
#include "e_I2C_protocol.h"
#include "e_init_port.h"
#include "e_al440b.h"

char _po3030k_line_conf[1281];

/*! This function setup the internal timing of the camera to match the 
 * zoom, color mode and interest area.
 * \warning If the common deniminator of both zoom factor is 4 or 2, part of the 
 * subsampling is done by the camera ( QQVGA = 4, QVGA = 2 ). This increase the 
 * framerate by respectivly 4 or 2. Moreover greyscale is twice faster than color mode.
 * \param sensor_x1 The X coordinate of the window's corner
 * \param sensor_y1 The Y coordinate of the window's corner
 * \param sensor_width the Width of the interest area, in FULL sampling scale
 * \param sensor_height The Height of the insterest area, in FULL sampling scale
 * \param zoom_fact_width The subsampling to apply for the window's Width
 * \param zoom_fact_height The subsampling to apply for the window's Height
 * \param color_mode The color mode in which the camera should be configured
 * \return Zero if the settings are correct, non-zero if an error occur 
 * \sa e_po3030k_write_cam_registers 
 */

int e_po3030k_config_cam(unsigned int sensor_x1,unsigned int sensor_y1,
				unsigned int sensor_width,unsigned int sensor_height,
				unsigned int zoom_fact_width,unsigned int zoom_fact_height,  
				int color_mode) {
	int pbp_h, pbp_w;
	int nb_cols;
	int real_zoom_h, real_zoom_w;
	int sampl_mode;
	int xearly=0;

	sensor_x1 += ARRAY_ORIGINE_X;
	sensor_y1 += ARRAY_ORIGINE_Y;
	/* testing if the mode is acceptable */
	if(zoom_fact_height < 1 || zoom_fact_width < 1)
		return -1;

	/* Check if the area is out of bound */
	if((sensor_x1 + sensor_width) > (ARRAY_ORIGINE_X + ARRAY_WIDTH)) 
		return -1;
	if((sensor_y1 + sensor_height) > (ARRAY_ORIGINE_Y + ARRAY_HEIGHT))
		return -1;
	
	/* Check if Sensor[Width|Height] is a multiple of Zoom */
	if(sensor_width % zoom_fact_width)
		return -1;
	if(sensor_height % zoom_fact_height)
		return -1;	

	/* Search the best subsampling available */
	if(!(zoom_fact_height%4)) {
		if(!(zoom_fact_width%4)) {
//			sampl_mode = MODE_QQVGA;
//			real_zoom_h = zoom_fact_height >> 2;
//			real_zoom_w = zoom_fact_width >> 2;
//			xearly = 8*IRQ_PIX_LAT;
			// for now, remove QQVGA mode
			sampl_mode = MODE_QVGA;
			real_zoom_h = zoom_fact_height >> 1;
			real_zoom_w = zoom_fact_width >> 1;
			xearly = 3*IRQ_PIX_LAT;	
		} else {
			if(!(zoom_fact_width%2)) {
				sampl_mode = MODE_QVGA;
				real_zoom_h = zoom_fact_height >> 1;
				real_zoom_w = zoom_fact_width >> 1;
				xearly = 3*IRQ_PIX_LAT;
			} else {
				sampl_mode = MODE_VGA;
				real_zoom_h = zoom_fact_height;
				real_zoom_w = zoom_fact_width;
				xearly = IRQ_PIX_LAT*3;
			}
		}
	} else if(!(zoom_fact_height%2)) {
		if(!(zoom_fact_width%2)) {
			sampl_mode = MODE_QVGA;
			real_zoom_h = zoom_fact_height >> 1;
			real_zoom_w = zoom_fact_width >> 1;
			xearly = 3*IRQ_PIX_LAT;
		} else {
			sampl_mode = MODE_VGA;
			real_zoom_h = zoom_fact_height;
			real_zoom_w = zoom_fact_width;
			xearly = IRQ_PIX_LAT*3;
		}
	} else {
		sampl_mode = MODE_VGA;
		real_zoom_w = zoom_fact_width;
		real_zoom_h = zoom_fact_height;
		xearly = IRQ_PIX_LAT*3;
	}

	// ----- set default values for omnicam, for testing -----------
//	xearly = IRQ_PIX_LAT*4;		// this value is set in the header file, depends on speed of pic!
//	sampl_mode = MODE_QQVGA;
//	real_zoom_h = zoom_fact_height>>2;
//	real_zoom_w = zoom_fact_width>>2;

	// -------------------------------------------------------------
	nb_cols = sensor_width / zoom_fact_width;
	pbp_w = real_zoom_w - 1;
	pbp_h = real_zoom_h - 1;
	/* set camera configuration */
	// ie. YUV, RGB656, etc.
	if(e_po3030k_set_color_mode(color_mode))
		return -1;
	if(e_po3030k_set_sampling_mode(sampl_mode)) 
		return -1;

	// sets the window size without extra pixels or rows at the end
	// ******* directly affects the size of the HSYNC signal, and # of pixels that is produced
	if(e_po3030k_set_wx(sensor_x1-xearly,sensor_x1+sensor_width)) 
//	if(e_po3030k_set_wx(sensor_x1-xearly,sensor_x1+sensor_width-1)) 
		return -1;
//	if(e_po3030k_set_wy(sensor_y1-yearly,sensor_y1+sensor_height))
	if(e_po3030k_set_wy(sensor_y1,sensor_y1+sensor_height))
		return -1;

	/* set timer configuration */
	if(e_po3030k_apply_timer_config(nb_cols,e_po3030k_get_bytes_per_pixel(color_mode),pbp_w))
		return -1;

	// set number of rows to skip when capturing an image into the FIFO
	a_set_blank_row_betw(pbp_h);		// = real_zoom_h - 1;

	return 0;
}

/*! Modify the interrupt configuration
 * \warning This is an internal function, use \a e_po3030k_config_cam
 * \param pixel_row The number of row to take
 * \param pixel_col The number of pixel to take each \a pixel_row
 * \param bpp The number of byte per pixel
 * \param pbp The number of pixel to ignore between each pixel
 * \param bbl The number of row to ignore between each line
 * \return Zero if OK, non-zero if the mode exceed internal data representation
 * \sa e_po3030k_get_bytes_per_pixel and e_po3030k_config_cam
 */
int e_po3030k_apply_timer_config(int pixel_col, int bpp, int pbp) {
	int i;
	int pos = 0;
	if(pixel_col * bpp * (1+pbp) + 1 > sizeof(_po3030k_line_conf))
		return -1;

	if (bpp==2)	{  // colour mode
		// skip the first byte...HSYNC interrupt misses first 3 PCLKs, means that the first byte
		// is half pixel #2 in colour mode
//		_po3030k_line_conf[pos++] = 0;
		
		// skip pixel 3 (2 bytes) as well, start image at pixel 4 to synchronise with black/white image
//		_po3030k_line_conf[pos++] = 0;
//		_po3030k_line_conf[pos++] = 0;
	}
	
	for(i = 0; i < pixel_col; i++) {
		int j;
		for(j = 0; j < bpp; j++) {
			_po3030k_line_conf[pos++] = 1;
		}
		for(j = 0; j < pbp*bpp; j++)  {
			_po3030k_line_conf[pos++] = 0;
		}
	}
//	if (bpp==2)
//		_po3030k_line_conf[pos-5] = 2;
//	else
  		/* flag to tell "line end here" */
		_po3030k_line_conf[pos-1] = 2;

	return 0;
}

/*! Return the number of bytes per pixel in the given color mode
 * \param color_mode The given color mode
 * \return The number of bytes per pixel in the given color mode
 */
int e_po3030k_get_bytes_per_pixel(int color_mode) {
	switch (color_mode) {
		case GREY_SCALE_MODE:
			return 1;
		case RGB_565_MODE:
			return 2;
		case YUV_MODE:
			return 2; /* YUV mode is not the best mode for subsampling */
	}
	/* UNKNOWN ! */
	return 1;
}

/*! Initialize the camera, must be called before any other function
 */
void e_po3030k_init_cam(void) {
	int i;
//	e_init_port();

	CAM_RESET=0;
	for(i=100;i;i--) __asm__ volatile ("nop");
	CAM_RESET=1;
	for(i=100;i;i--) __asm__ volatile ("nop");
	/* enable interrupt nesting */
	INTCON1bits.NSTDIS = 0;
	/* set a higher priority on camera's interrupts */
//	IPC5 = (IPC5 & 0xF00F) + 0x0660;

	// set ISP Control 3 register:  (Sepia Color Diable, VSYNC not inverted, HSYNC not inverted, All region HSYNC)
	e_po3030k_set_register(0x4F,0x0A);

	// set the rising and falling time of VSYNC - row start,row stop,column start
	e_po3030k_set_vsync(ARRAY_ORIGINE_Y,ARRAY_ORIGINE_Y + ARRAY_HEIGHT, ARRAY_ORIGINE_X-40); // set VSYNC to bigger than frame to make sure there is no latency on the HSYNC interrupt
		
	// cannot run at highest speed, pixel clock is not synchronised with dsPIC,
	// thus cannot be sure of interrupt latency (how many pixels are skipped by HSYNC interrupt)
	e_po3030k_set_speed(SPEED_4);  		
}
	
